<?php

// ------------------------------------------------------------
// LOAD CONFIG
// ------------------------------------------------------------
$config = require __DIR__ . '/config.php';

$BASEDIR = realpath($config['basedir']);
if (!$BASEDIR) {
    die("Invalid or non-existent basedir in config.php\n");
}


// ------------------------------------------------------------
// HELPER FUNCTIONS
// ------------------------------------------------------------

function safe_join($base, $path) {
    // Normalize slashes and strip any obvious traversal
    $path = str_replace(['..', '\\'], ['', '/'], $path);

    $full = realpath($base . DIRECTORY_SEPARATOR . $path);
    if (!$full || strpos($full, $base) !== 0) {
        die("Security error: invalid path traversal attempt\n");
    }
    return $full;
}

function read_json($path) {
    if (!file_exists($path)) return null;
    return json_decode(file_get_contents($path), true);
}

function h($str) {
    return htmlspecialchars($str ?? '', ENT_QUOTES, 'UTF-8');
}

function link_to($params) {
    return '?' . http_build_query($params);
}

function render_header($title) {
    echo "<!DOCTYPE html><html><head><meta charset='utf-8'>";
    echo "<title>" . h($title) . "</title>";
    echo "<style>
        body { font-family: sans-serif; margin: 20px; }
        .workspace { margin-bottom: 20px; }
        .board-list { margin-left: 20px; }
        .columns { display: flex; gap: 20px; align-items: flex-start; }
        .column { background: #f0f0f0; padding: 10px; border-radius: 6px; min-width: 250px; }
        .card { background: white; padding: 8px; margin-bottom: 8px; border-radius: 4px; border: 1px solid #ccc; }
        .card a { text-decoration: none; color: #333; }
        .comment { background: #fafafa; padding: 8px; margin-bottom: 10px; border-left: 3px solid #999; }
        .attachment { font-size: 0.9em; color: #555; }
        a { color: #0645AD; }
        form.delete-form { margin-top: 20px; }
        form.delete-form button { background: #c00; color: white; border: none; padding: 8px 12px; border-radius: 4px; cursor: pointer; }
        form.delete-form button:hover { background: #a00; }
    </style>";
    echo "</head><body>";
    echo "<h1>" . h($title) . "</h1>";
}

function render_footer() {
    echo "</body></html>";
}

function ensure_dir($dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0777, true);
    }
}

function list_export_dirs($base) {
    $dirs = glob($base . "/export_*", GLOB_ONLYDIR);
    sort($dirs);
    return $dirs;
}


// ------------------------------------------------------------
// EXPORT DIRECTORY SELECTION
// ------------------------------------------------------------

$exportDirs = list_export_dirs($BASEDIR);
$selectedExport = $_REQUEST['export'] ?? null; // GET or POST

if (!$selectedExport) {
    if (count($exportDirs) === 0) {
        render_header("No exports found");
        echo "<p>No export_* directories found under basedir.</p>";
        render_footer();
        exit;
    }

    if (count($exportDirs) === 1) {
        $selectedExport = basename($exportDirs[0]);
    } else {
        render_header("Select Export Directory");
        echo "<p>Multiple export directories found. Please choose one:</p><ul>";

        foreach ($exportDirs as $dir) {
            echo "<li><a href='" . link_to(['export' => basename($dir)]) . "'>" . h(basename($dir)) . "</a></li>";
        }

        echo "</ul>";
        render_footer();
        exit;
    }
}

// Resolve selected export directory safely (must exist)
$exportDir = safe_join($BASEDIR, $selectedExport);


// ------------------------------------------------------------
// ROUTING VARIABLES
// ------------------------------------------------------------

$workspace = $_REQUEST['ws'] ?? null;
$board     = $_REQUEST['board'] ?? null;
$card      = $_REQUEST['card'] ?? null;


// ------------------------------------------------------------
// DELETE CARD → MOVE TO TRASH (T3 layout)
// ------------------------------------------------------------
// <basedir>/export_YYYY.../<workspace>/<board>/trash/cards/<cardId>/

if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'delete_card') {
    if ($workspace && $board && $card) {
        // Resolve safe base dirs that already exist
        $boardDir = safe_join($exportDir, "$workspace/$board");
        $cardDir  = safe_join($exportDir, "$workspace/$board/cards/$card");

        if (!is_dir($cardDir)) {
            render_header("Card not found");
            echo "<p>Card directory not found. It may have already been deleted.</p>";
            echo "<p><a href='" . link_to([
                'export' => basename($exportDir),
                'ws' => $workspace,
                'board' => $board
            ]) . "'>Back to board</a></p>";
            render_footer();
            exit;
        }

        // Build trash base from the already-safe $boardDir
        $trashBase = $boardDir . '/trash/cards';
        ensure_dir($trashBase);

        $trashCardDir = $trashBase . DIRECTORY_SEPARATOR . $card;
        if (is_dir($trashCardDir)) {
            $trashCardDir .= '_' . date('YmdHis');
        }

        if (!rename($cardDir, $trashCardDir)) {
            render_header("Delete failed");
            echo "<p>Failed to move card to trash.</p>";
            echo "<p><a href='" . link_to([
                'export' => basename($exportDir),
                'ws' => $workspace,
                'board' => $board,
                'card' => $card
            ]) . "'>Back to card</a></p>";
            render_footer();
            exit;
        }

        header("Location: " . link_to([
            'export' => basename($exportDir),
            'ws' => $workspace,
            'board' => $board
        ]));
        exit;
    }
}


// ------------------------------------------------------------
// PAGE 1: Workspaces + Boards
// ------------------------------------------------------------
if (!$workspace) {
    render_header("Trello Export Viewer");

    echo "<p>Using export directory: <strong>" . h(basename($exportDir)) . "</strong></p>";

    $workspaces = glob("$exportDir/*", GLOB_ONLYDIR);

    foreach ($workspaces as $wsPath) {
        $wsName = basename($wsPath);
        if ($wsName === 'trash') continue;

        echo "<div class='workspace'>";
        echo "<h2>" . h($wsName) . "</h2>";

        $boards = glob("$wsPath/*", GLOB_ONLYDIR);
        echo "<ul class='board-list'>";
        foreach ($boards as $boardPath) {
            $boardName = basename($boardPath);
            if ($boardName === 'trash') continue;

            echo "<li><a href='" . link_to([
                'export' => basename($exportDir),
                'ws' => $wsName,
                'board' => $boardName
            ]) . "'>" . h($boardName) . "</a></li>";
        }
        echo "</ul></div>";
    }

    render_footer();
    exit;
}


// ------------------------------------------------------------
// PAGE 2: Board view (lists + cards)
// ------------------------------------------------------------
if ($workspace && $board && !$card) {
    $boardDir = safe_join($exportDir, "$workspace/$board");

    render_header("Board: $board");

    echo "<p><a href='" . link_to(['export' => basename($exportDir)]) . "'>← Back to workspaces</a></p>";

    $listsDir = "$boardDir/lists";
    $cardsDir = "$boardDir/cards";

    // Load + sort lists by pos
    $listFiles = glob("$listsDir/*.json");
    $lists = [];
    foreach ($listFiles as $lf) {
        $list = read_json($lf);
        if ($list) $lists[] = $list;
    }

    usort($lists, function($a, $b) {
        return ($a['pos'] ?? 0) <=> ($b['pos'] ?? 0);
    });

    echo "<div class='columns'>";

    foreach ($lists as $list) {
        $listName = $list['name'] ?? 'Unnamed list';
        $listId   = $list['id'];

        echo "<div class='column'>";
        echo "<h3>" . h($listName) . "</h3>";

        // Load + sort cards belonging to this list
        $cardDirs = glob("$cardsDir/*", GLOB_ONLYDIR);
        $cards = [];

        foreach ($cardDirs as $cardDir) {
            $cardJson = read_json("$cardDir/card.json");
            if ($cardJson && ($cardJson['idList'] ?? null) === $listId) {
                $cards[] = [
                    'data' => $cardJson,
                    'dir'  => $cardDir
                ];
            }
        }

        usort($cards, function($a, $b) {
            return ($a['data']['pos'] ?? 0) <=> ($b['data']['pos'] ?? 0);
        });

        foreach ($cards as $c) {
            $cardJson = $c['data'];
            $cardId   = basename($c['dir']);

            echo "<div class='card'>";
            echo "<a href='" . link_to([
                'export' => basename($exportDir),
                'ws' => $workspace,
                'board' => $board,
                'card' => $cardId
            ]) . "'>" . h($cardJson['name'] ?? 'Untitled card') . "</a>";
            echo "</div>";
        }

        echo "</div>";
    }

    echo "</div>";

    render_footer();
    exit;
}


// ------------------------------------------------------------
// PAGE 3: Card view (title, description, comments + delete)
// ------------------------------------------------------------
if ($workspace && $board && $card) {
    $cardDir = safe_join($exportDir, "$workspace/$board/cards/$card");

    if (!is_dir($cardDir)) {
        render_header("Card not found");
        echo "<p>Card directory not found. It may have been moved to trash.</p>";
        echo "<p><a href='" . link_to([
            'export' => basename($exportDir),
            'ws' => $workspace,
            'board' => $board
        ]) . "'>← Back to board</a></p>";
        render_footer();
        exit;
    }

    $cardJson = read_json("$cardDir/card.json");
    $comments = read_json("$cardDir/comments.json");
    $attachments = read_json("$cardDir/attachments.json");

    $title = $cardJson['name'] ?? 'Untitled card';
    $desc  = $cardJson['desc'] ?? '';

    render_header("Card: $title");

    echo "<p><a href='" . link_to([
        'export' => basename($exportDir),
        'ws' => $workspace,
        'board' => $board
    ]) . "'>← Back to board</a></p>";

    echo "<h2>" . h($title) . "</h2>";
    echo "<p>" . nl2br(h($desc)) . "</p>";

    // Attachments metadata
    if ($attachments) {
        echo "<h3>Attachments</h3>";
        foreach ($attachments as $att) {
            echo "<div class='attachment'>";
            echo "<strong>" . h($att['name']) . "</strong><br>";
            echo "Uploaded: " . ($att['isUpload'] ? "Yes" : "No") . "<br>";
            echo "MIME: " . h($att['mimeType']) . "<br>";
            echo "Size: " . h($att['size']) . "<br>";
            echo "Download status: " . h($att['http_status']) . "<br>";
            echo "</div><br>";
        }
    }

    // Comments
    echo "<h3>Comments</h3>";
    if ($comments) {
        foreach ($comments as $c) {
            $text = $c['data']['text'] ?? '';
            $author = $c['memberCreator']['fullName'] ?? 'Unknown';
            $date = $c['date'] ?? '';

            echo "<div class='comment'>";
            echo "<strong>" . h($author) . "</strong> <em>(" . h($date) . ")</em><br>";
            echo nl2br(h($text));
            echo "</div>";
        }
    } else {
        echo "<p>No comments.</p>";
    }

    // Delete (move to trash) form
    echo "<h3>Danger zone</h3>";
    echo "<form method='post' class='delete-form' onsubmit=\"return confirm('Really move this card to trash?');\">";
    echo "<input type='hidden' name='export' value='" . h(basename($exportDir)) . "'>";
    echo "<input type='hidden' name='ws' value='" . h($workspace) . "'>";
    echo "<input type='hidden' name='board' value='" . h($board) . "'>";
    echo "<input type='hidden' name='card' value='" . h($card) . "'>";
    echo "<input type='hidden' name='action' value='delete_card'>";
    echo "<button type='submit'>Move this card to trash</button>";
    echo "</form>";

    render_footer();
    exit;
}

