<?php

// ------------------------------------------------------------
// LOAD CONFIG
// ------------------------------------------------------------
$config = require __DIR__ . '/config.php';

$API_KEY  = $config['key'];
$TOKEN    = $config['token'];
$BASE     = "https://api.trello.com/1";

$BASEDIR  = realpath($config['basedir']);
if (!$BASEDIR) {
    die("Invalid or non-existant basedir in config.php\n");
}

// Ensure basedir exists
if (!is_dir($BASEDIR)) {
    mkdir($BASEDIR, 0777, true);
}

// Create timestamped export directory
$timestamp = date('Ymd\THis');
$EXPORT_ROOT = $BASEDIR . DIRECTORY_SEPARATOR . "export_$timestamp";


// ------------------------------------------------------------
// HELPERS
// ------------------------------------------------------------

function safe_join($base, $path) {
    $full = realpath($base . DIRECTORY_SEPARATOR . $path);
    if (!$full || strpos($full, $base) !== 0) {
        die("Security error: invalid path traversal attempt\n");
    }
    return $full;
}

function trello_get($path, $params = []) {
    global $API_KEY, $TOKEN, $BASE;

    $params['key']   = $API_KEY;
    $params['token'] = $TOKEN;

    $url = $BASE . $path . '?' . http_build_query($params);

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL            => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
    ]);

    $response = curl_exec($ch);
    $status   = curl_getinfo($ch, CURLINFO_HTTP_CODE);

    if ($response === false) {
        $err = curl_error($ch);
        curl_close($ch);
        throw new Exception("cURL GET failed: $url ($err)");
    }

    curl_close($ch);

    if ($status < 200 || $status >= 300) {
        throw new Exception("HTTP $status for $url: $response");
    }

    return json_decode($response, true);
}

function safe($name) {
    $name = preg_replace('/[^a-zA-Z0-9 _-]/', '', $name);
    return trim($name) !== '' ? trim($name) : 'unnamed';
}

function ensure_dir($dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0777, true);
    }
}

function save_json($path, $data) {
    ensure_dir(dirname($path));
    file_put_contents($path, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
}

/**
 * Always use the API download endpoint, never the UI URL.
 */
function api_download_url($cardId, $attachmentId) {
    global $API_KEY, $TOKEN;

    return "https://api.trello.com/1/cards/$cardId/attachments/$attachmentId/download"
         . "?key=$API_KEY&token=$TOKEN";
}


/**
 * Download attachments in parallel using curl_multi.
 */
function download_attachments_parallel($cardId, array &$attachmentInfo, $targetDir) {
    ensure_dir($targetDir);

    $mh      = curl_multi_init();
    $handles = [];

    foreach ($attachmentInfo as $index => $info) {

        if (empty($info['isUpload']) || $info['isUpload'] !== true) {
            echo "      Skipping non-upload attachment: {$info['name']}\n";
            continue;
        }

        $url = $info['api_url'];

        $name = safe($info['name']);
        $ext  = pathinfo($info['name'], PATHINFO_EXTENSION);
        $filename = $ext ? "$name.$ext" : $name;

        $filepath = $targetDir . DIRECTORY_SEPARATOR . $filename;

        echo "      Queuing file: $filename\n";

        $fh = fopen($filepath, 'wb');
        if (!$fh) {
            echo "      Failed to open file for writing: $filepath\n";
            continue;
        }

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL            => $url,
            CURLOPT_FILE           => $fh,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_SSL_VERIFYHOST => 2,
            CURLOPT_FAILONERROR    => false,
        ]);

        curl_multi_add_handle($mh, $ch);

        $handles[] = [
            'ch'   => $ch,
            'fh'   => $fh,
            'file' => $filepath,
            'url'  => $url,
            'index'=> $index,
        ];
    }

    // Execute all queued downloads
    $running = null;
    do {
        curl_multi_exec($mh, $running);
        curl_multi_select($mh, 1.0);
    } while ($running > 0);

    // Cleanup
    foreach ($handles as $h) {
        $ch = $h['ch'];
        $fh = $h['fh'];
        $idx = $h['index'];

        $errNo  = curl_errno($ch);
        $errMsg = curl_error($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        curl_multi_remove_handle($mh, $ch);
        curl_close($ch);
        fclose($fh);

        $attachmentInfo[$idx]['http_status'] = $code;
        $attachmentInfo[$idx]['downloaded']  = ($code === 200);

        if ($code === 200) {
            echo "      Downloaded OK: {$attachmentInfo[$idx]['name']}\n";
        } else {
            echo "      ERROR downloading {$attachmentInfo[$idx]['name']} (HTTP $code)\n";
        }
    }

    curl_multi_close($mh);
}


// ------------------------------------------------------------
// MAIN EXPORT LOGIC
// ------------------------------------------------------------

function export_all() {
    global $EXPORT_ROOT;

    echo "Export directory: $EXPORT_ROOT\n";
    ensure_dir($EXPORT_ROOT);

    echo "Fetching workspaces...\n";

    $workspaces = trello_get("/members/me/organizations");

    foreach ($workspaces as $ws) {
        $wsName = safe($ws['displayName'] ?? $ws['name'] ?? 'workspace');
        $wsDir  = $EXPORT_ROOT . DIRECTORY_SEPARATOR . $wsName;
        ensure_dir($wsDir);

        echo "Workspace: $wsName\n";

        $boards = trello_get("/organizations/{$ws['id']}/boards");

        foreach ($boards as $board) {
            $boardName = safe($board['name'] ?? 'board');
            $boardDir  = "$wsDir/$boardName";
            ensure_dir($boardDir);

            echo "  Board: $boardName\n";

            save_json("$boardDir/board.json", $board);

            // Lists
            $lists = trello_get("/boards/{$board['id']}/lists");
            $listsDir = "$boardDir/lists";
            ensure_dir($listsDir);

            foreach ($lists as $list) {
                $listName = safe($list['name'] ?? 'list');
                save_json("$listsDir/$listName.json", $list);
            }

            // Cards
            $cards = trello_get("/boards/{$board['id']}/cards");
            $cardsDir = "$boardDir/cards";
            ensure_dir($cardsDir);

            foreach ($cards as $card) {
                $cardId  = $card['id'];
                $cardDir = "$cardsDir/$cardId";
                ensure_dir($cardDir);

                echo "    Card: " . ($card['name'] ?? 'card') . "\n";

                save_json("$cardDir/card.json", $card);

                // Comments
                $comments = trello_get("/cards/{$cardId}/actions", [
                    'filter' => 'commentCard',
                ]);
                save_json("$cardDir/comments.json", $comments);

                // Attachments
                $attachments = trello_get("/cards/{$cardId}/attachments");
                $attachDir   = "$cardDir/attachments";

                // Build metadata
                $attachmentInfo = [];
                foreach ($attachments as $att) {
                    $attachmentInfo[] = [
                        'id'          => $att['id'],
                        'name'        => $att['name'],
                        'isUpload'    => $att['isUpload'] ?? null,
                        'mimeType'    => $att['mimeType'] ?? null,
                        'size'        => $att['bytes'] ?? null,
                        'ui_url'      => $att['url'],
                        'api_url'     => api_download_url($cardId, $att['id']),
                        'downloaded'  => false,
                        'http_status' => null,
                    ];
                }

                // Save metadata before downloads
                save_json("$cardDir/attachments.json", $attachmentInfo);

                // Attempt downloads
                if (!empty($attachmentInfo)) {
                    //This doesn't work for now
                    //download_attachments_parallel($cardId, $attachmentInfo, $attachDir);
                }

                // Save updated metadata (with status codes)
                save_json("$cardDir/attachments.json", $attachmentInfo);
            }
        }
    }

    echo "Done.\n";
}

export_all();

