<?php

$config = require __DIR__ . '/config.php';

$API_KEY  = $config['key'];
$TOKEN    = $config['token'];

$BASEDIR = realpath($config['basedir']);
if (!$BASEDIR) die("Invalid basedir\n");

function safe_join($base, $path) {
    $full = realpath($base . DIRECTORY_SEPARATOR . $path);
    if (!$full || strpos($full, $base) !== 0) {
        die("Security error: invalid path traversal attempt\n");
    }
    return $full;
}

function api_download_url($cardId, $attachmentId) {
    global $API_KEY, $TOKEN;
    return "https://api.trello.com/1/cards/$cardId/attachments/$attachmentId/download"
         . "?key=$API_KEY&token=$TOKEN";
}

function safe($name) {
    return preg_replace('/[^a-zA-Z0-9 _-]/', '', $name);
}

function retry_download($cardDir, &$info) {
    $cardId = basename($cardDir);
    $url    = $info['api_url'];

    $name = safe($info['name']);
    $ext  = pathinfo($info['name'], PATHINFO_EXTENSION);
    $filename = $ext ? "$name.$ext" : $name;

    $filepath = "$cardDir/attachments/$filename";

    echo "  Retrying: $filename\n";

    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL            => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
    ]);

    $data = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    $info['http_status'] = $code;
    $info['downloaded']  = ($code === 200);

    if ($code === 200) {
        file_put_contents($filepath, $data);
        echo "    OK\n";
    } else {
        echo "    FAILED ($code)\n";
    }
}

function scan_export($root, $force = false) {
    $rii = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($root));

    foreach ($rii as $file) {
        if ($file->getFilename() !== 'attachments.json') continue;

        $jsonPath = $file->getPathname();
        $cardDir  = dirname($jsonPath);

        echo "\nProcessing: $jsonPath\n";

        $data = json_decode(file_get_contents($jsonPath), true);

        foreach ($data as &$att) {

            if (empty($att['isUpload']) || $att['isUpload'] !== true) {
                continue;
            }

            $name = safe($att['name']);
            $ext  = pathinfo($att['name'], PATHINFO_EXTENSION);
            $filename = $ext ? "$name.$ext" : $name;

            $filepath = "$cardDir/attachments/$filename";
            $fileExists = file_exists($filepath);

            $shouldRetry = $force || (!$att['downloaded'] && !$fileExists);

            if ($shouldRetry) {
                retry_download($cardDir, $att);
            }
        }

        file_put_contents($jsonPath, json_encode($data, JSON_PRETTY_PRINT));
    }
}

$exportDir = $argv[1] ?? null;
$force     = in_array('--force', $argv, true);

if (!$exportDir) {
    die("Usage: php download_later.php <export_directory> [--force]\n");
}

$exportDir = safe_join($BASEDIR, $exportDir);

scan_export($exportDir, $force);

echo "\nDone.\n";

